<?php
/**
 * Plugin Name: One Click Clear Comments / 一键清空评论 / ワンクリックコメント削除
 * Plugin URI: https://yoursite.com/
 * Description: Clear comments by category with multilingual support / 分类清空评论，支持多语言 / カテゴリ別にコメントを削除、多言語対応
 * Version: 1.2.0
 * Author: Your Name
 * License: GPL v2 or later
 * Text Domain: one-click-clear-comments
 * Domain Path: /languages
 */

// 防止直接访问
if (!defined('ABSPATH')) {
    exit;
}

// 加载插件语言文件
add_action('plugins_loaded', 'occ_load_textdomain');

function occ_load_textdomain() {
    load_plugin_textdomain('one-click-clear-comments', false, dirname(plugin_basename(__FILE__)) . '/languages');
}

// 获取当前语言
function occ_get_current_language() {
    $locale = get_locale();
    if (strpos($locale, 'ja') === 0) {
        return 'ja';
    } elseif (strpos($locale, 'zh_CN') === 0) {
        return 'zh_CN';
    }
    return 'en';
}

// 翻译函数
function occ_t($key) {
    $lang = occ_get_current_language();
    $translations = array(
        'menu_title' => array(
            'en' => 'Clear Comments',
            'zh_CN' => '清空评论',
            'ja' => 'コメント削除'
        ),
        'page_title' => array(
            'en' => 'One Click Clear Comments',
            'zh_CN' => '一键清空评论',
            'ja' => 'ワンクリックコメント削除'
        ),
        'comment_stats' => array(
            'en' => 'Current Comment Statistics',
            'zh_CN' => '当前评论统计',
            'ja' => '現在のコメント統計'
        ),
        'total_comments' => array(
            'en' => 'Total Comments',
            'zh_CN' => '总评论数',
            'ja' => '総コメント数'
        ),
        'approved' => array(
            'en' => 'Approved',
            'zh_CN' => '已批准',
            'ja' => '承認済み'
        ),
        'pending' => array(
            'en' => 'Pending',
            'zh_CN' => '待审核',
            'ja' => '承認待ち'
        ),
        'spam' => array(
            'en' => 'Spam',
            'zh_CN' => '垃圾评论',
            'ja' => 'スパム'
        ),
        'trash' => array(
            'en' => 'Trash',
            'zh_CN' => '回收站',
            'ja' => 'ゴミ箱'
        ),
        'site_comments_control' => array(
            'en' => 'Site-wide Comments Control',
            'zh_CN' => '整站评论控制',
            'ja' => 'サイト全体のコメント制御'
        ),
        'comment_status' => array(
            'en' => 'Comment Status',
            'zh_CN' => '评论状态',
            'ja' => 'コメントステータス'
        ),
        'disable_comments' => array(
            'en' => 'Disable site-wide comments',
            'zh_CN' => '禁用整站评论功能',
            'ja' => 'サイト全体のコメント機能を無効化'
        ),
        'comments_disabled' => array(
            'en' => '✖ Comments disabled - All comments are closed',
            'zh_CN' => '✖ 评论功能已禁用 - 所有文章和页面的评论功能都已关闭',
            'ja' => '✖ コメント無効 - すべてのコメント機能が閉じられています'
        ),
        'comments_enabled' => array(
            'en' => '✓ Comments enabled - Visitors can leave comments',
            'zh_CN' => '✓ 评论功能已启用 - 访客可以发表评论',
            'ja' => '✓ コメント有効 - 訪問者はコメントを投稿できます'
        ),
        'save_settings' => array(
            'en' => 'Save Settings',
            'zh_CN' => '保存设置',
            'ja' => '設定を保存'
        ),
        'clear_by_type' => array(
            'en' => 'Clear by Type',
            'zh_CN' => '分类清空',
            'ja' => 'タイプ別削除'
        ),
        'select_type' => array(
            'en' => 'Select comment type to clear:',
            'zh_CN' => '选择要清空的评论类型：',
            'ja' => '削除するコメントタイプを選択：'
        ),
        'clear_pending' => array(
            'en' => 'Clear Pending',
            'zh_CN' => '清空待审核',
            'ja' => '承認待ちを削除'
        ),
        'clear_spam' => array(
            'en' => 'Clear Spam',
            'zh_CN' => '清空垃圾评论',
            'ja' => 'スパムを削除'
        ),
        'clear_trash' => array(
            'en' => 'Clear Trash',
            'zh_CN' => '清空回收站',
            'ja' => 'ゴミ箱を空にする'
        ),
        'clear_unapproved' => array(
            'en' => 'Clear All Unapproved Comments',
            'zh_CN' => '一键清空所有未批准评论',
            'ja' => '未承認コメントを一括削除'
        ),
        'clear_unapproved_desc' => array(
            'en' => 'This will clear all pending, spam and trash comments, but keep approved comments.',
            'zh_CN' => '此操作将清空所有待审核、垃圾评论和回收站中的评论，但保留已批准的评论。',
            'ja' => 'この操作は承認待ち、スパム、ゴミ箱のコメントをすべて削除しますが、承認済みのコメントは保持されます。'
        ),
        'clear_all_unapproved' => array(
            'en' => 'Clear All Unapproved Comments',
            'zh_CN' => '清空所有未批准评论',
            'ja' => '未承認コメントをすべて削除'
        ),
        'danger_clear_all' => array(
            'en' => '⚠️ Danger: Clear All Comments',
            'zh_CN' => '⚠️ 危险操作：清空所有评论',
            'ja' => '⚠️ 危険：すべてのコメントを削除'
        ),
        'warning_all' => array(
            'en' => 'Warning: This will permanently delete ALL comments, including approved ones. This cannot be undone!',
            'zh_CN' => '警告：此操作将永久删除所有评论，包括已批准的评论。此操作不可撤销！',
            'ja' => '警告：この操作はすべてのコメント（承認済みを含む）を永久に削除します。この操作は取り消せません！'
        ),
        'confirm_delete_all' => array(
            'en' => 'I understand this will permanently delete all',
            'zh_CN' => '我理解此操作将永久删除所有',
            'ja' => 'すべてを永久に削除することを理解しています'
        ),
        'comments' => array(
            'en' => 'comments',
            'zh_CN' => '条评论',
            'ja' => '件のコメント'
        ),
        'clear_all_comments' => array(
            'en' => 'Clear All Comments',
            'zh_CN' => '清空所有评论',
            'ja' => 'すべてのコメントを削除'
        ),
        'instructions' => array(
            'en' => 'Instructions',
            'zh_CN' => '使用说明',
            'ja' => '使用方法'
        ),
        'instruction_1' => array(
            'en' => 'Clear by Type: Clear pending, spam or trash comments separately',
            'zh_CN' => '分类清空：可以单独清空待审核、垃圾评论或回收站中的评论',
            'ja' => 'タイプ別削除：承認待ち、スパム、ゴミ箱のコメントを個別に削除できます'
        ),
        'instruction_2' => array(
            'en' => 'Clear Unapproved: Clear all pending, spam and trash at once, keep approved',
            'zh_CN' => '清空未批准：一次性清空所有待审核、垃圾评论和回收站评论，保留已批准评论',
            'ja' => '未承認削除：承認待ち、スパム、ゴミ箱を一括削除、承認済みは保持'
        ),
        'instruction_3' => array(
            'en' => 'Clear All: Delete all comments including approved ones',
            'zh_CN' => '清空所有：删除网站上的所有评论，包括已批准的评论',
            'ja' => 'すべて削除：承認済みを含むすべてのコメントを削除'
        ),
        'instruction_4' => array(
            'en' => 'Comment counts will be updated automatically after deletion',
            'zh_CN' => '删除后，相关文章的评论计数会自动更新',
            'ja' => '削除後、関連記事のコメント数は自動的に更新されます'
        ),
        'instruction_5' => array(
            'en' => 'Please backup your database before any operation',
            'zh_CN' => '建议在执行操作前先备份数据库',
            'ja' => '操作前にデータベースのバックアップを推奨します'
        ),
        'success_settings' => array(
            'en' => 'Settings updated successfully!',
            'zh_CN' => '评论设置已更新！',
            'ja' => '設定が更新されました！'
        ),
        'success_pending' => array(
            'en' => 'All pending comments cleared!',
            'zh_CN' => '所有待审核评论已清空！',
            'ja' => 'すべての承認待ちコメントが削除されました！'
        ),
        'success_spam' => array(
            'en' => 'All spam comments cleared!',
            'zh_CN' => '所有垃圾评论已清空！',
            'ja' => 'すべてのスパムコメントが削除されました！'
        ),
        'success_trash' => array(
            'en' => 'Trash cleared!',
            'zh_CN' => '回收站中的评论已清空！',
            'ja' => 'ゴミ箱が空になりました！'
        ),
        'success_unapproved' => array(
            'en' => 'All unapproved comments cleared!',
            'zh_CN' => '所有待审核、垃圾评论和回收站评论已清空！',
            'ja' => 'すべての未承認コメントが削除されました！'
        ),
        'success_all' => array(
            'en' => 'All comments cleared successfully!',
            'zh_CN' => '所有评论已成功清空！',
            'ja' => 'すべてのコメントが正常に削除されました！'
        ),
        'error' => array(
            'en' => 'Operation failed, please try again.',
            'zh_CN' => '操作失败，请重试。',
            'ja' => '操作に失敗しました。もう一度お試しください。'
        ),
        'confirm_pending' => array(
            'en' => 'Are you sure you want to clear all pending comments?',
            'zh_CN' => '确定要清空所有待审核评论吗？',
            'ja' => 'すべての承認待ちコメントを削除してもよろしいですか？'
        ),
        'confirm_spam' => array(
            'en' => 'Are you sure you want to clear all spam comments?',
            'zh_CN' => '确定要清空所有垃圾评论吗？',
            'ja' => 'すべてのスパムコメントを削除してもよろしいですか？'
        ),
        'confirm_trash' => array(
            'en' => 'Are you sure you want to empty the trash?',
            'zh_CN' => '确定要清空回收站中的评论吗？',
            'ja' => 'ゴミ箱を空にしてもよろしいですか？'
        ),
        'confirm_unapproved' => array(
            'en' => 'Are you sure you want to clear all pending, spam and trash comments?',
            'zh_CN' => '确定要清空所有待审核、垃圾评论和回收站评论吗？',
            'ja' => 'すべての承認待ち、スパム、ゴミ箱のコメントを削除してもよろしいですか？'
        ),
        'will_delete' => array(
            'en' => 'This will delete',
            'zh_CN' => '将删除',
            'ja' => '削除されるコメント数：'
        ),
        'final_confirm' => array(
            'en' => 'Final confirmation: Do you really want to delete all',
            'zh_CN' => '最后确认：您真的要删除所有',
            'ja' => '最終確認：本当にすべての'
        ),
        'cannot_undo' => array(
            'en' => 'comments? This action cannot be undone!',
            'zh_CN' => '条评论吗？此操作将删除所有评论（包括已批准的），且不可恢复！',
            'ja' => '件のコメントを削除しますか？この操作は取り消せません！'
        )
    );
    
    return isset($translations[$key][$lang]) ? $translations[$key][$lang] : $translations[$key]['en'];
}

// 添加管理菜单
add_action('admin_menu', 'occ_add_admin_menu');

function occ_add_admin_menu() {
    add_menu_page(
        occ_t('page_title'),
        occ_t('menu_title'),
        'manage_options',
        'one-click-clear-comments',
        'occ_admin_page',
        'dashicons-trash',
        80
    );
}

// 管理页面
function occ_admin_page() {
    // 检查用户权限
    if (!current_user_can('manage_options')) {
        return;
    }

    // 处理禁用评论开关
    if (isset($_POST['toggle_comments']) && check_admin_referer('occ_toggle_comments_nonce')) {
        $disable_comments = isset($_POST['disable_comments']) ? 'yes' : 'no';
        update_option('occ_disable_all_comments', $disable_comments);
        echo '<div class="notice notice-success is-dismissible"><p>' . occ_t('success_settings') . '</p></div>';
    }

    // 处理表单提交
    if (isset($_POST['action']) && check_admin_referer('occ_clear_comments_nonce')) {
        $action = sanitize_text_field($_POST['action']);
        $result = false;
        $message = '';
        
        switch ($action) {
            case 'clear_pending':
                $result = occ_clear_comments_by_status('0');
                $message = occ_t('success_pending');
                break;
            case 'clear_spam':
                $result = occ_clear_comments_by_status('spam');
                $message = occ_t('success_spam');
                break;
            case 'clear_trash':
                $result = occ_clear_comments_by_status('trash');
                $message = occ_t('success_trash');
                break;
            case 'clear_unwanted':
                $result = occ_clear_unwanted_comments();
                $message = occ_t('success_unapproved');
                break;
            case 'clear_all':
                $result = occ_clear_all_comments();
                $message = occ_t('success_all');
                break;
        }
        
        if ($result) {
            echo '<div class="notice notice-success is-dismissible"><p>' . $message . '</p></div>';
        } elseif ($_POST['action']) {
            echo '<div class="notice notice-error is-dismissible"><p>' . occ_t('error') . '</p></div>';
        }
    }

    // 获取当前评论统计
    $comment_count = wp_count_comments();
    $total_comments = $comment_count->total_comments;
    $comments_disabled = get_option('occ_disable_all_comments', 'no');
    ?>
    <div class="wrap">
        <h1><?php echo occ_t('page_title'); ?></h1>
        
        <!-- 整站评论开关 -->
        <div class="card" style="max-width: 600px; margin-top: 20px; <?php echo $comments_disabled === 'yes' ? 'border-color: #d63638;' : 'border-color: #00a32a;'; ?>">
            <h2><?php echo occ_t('site_comments_control'); ?></h2>
            <form method="post" action="">
                <?php wp_nonce_field('occ_toggle_comments_nonce'); ?>
                <table class="form-table">
                    <tr>
                        <th scope="row"><?php echo occ_t('comment_status'); ?></th>
                        <td>
                            <label>
                                <input type="checkbox" name="disable_comments" value="1" <?php checked($comments_disabled, 'yes'); ?>>
                                <strong><?php echo occ_t('disable_comments'); ?></strong>
                            </label>
                            <p class="description">
                                <?php if ($comments_disabled === 'yes'): ?>
                                    <span style="color: #d63638;"><?php echo occ_t('comments_disabled'); ?></span>
                                <?php else: ?>
                                    <span style="color: #00a32a;"><?php echo occ_t('comments_enabled'); ?></span>
                                <?php endif; ?>
                            </p>
                        </td>
                    </tr>
                </table>
                <p>
                    <input type="submit" name="toggle_comments" class="button button-primary" value="<?php echo occ_t('save_settings'); ?>">
                </p>
            </form>
        </div>
        
        <div class="card" style="max-width: 600px; margin-top: 20px;">
            <h2><?php echo occ_t('comment_stats'); ?></h2>
            <p><?php echo occ_t('total_comments'); ?>：<strong><?php echo number_format($total_comments); ?></strong></p>
            <ul>
                <li><?php echo occ_t('approved'); ?>：<?php echo number_format($comment_count->approved); ?></li>
                <li><?php echo occ_t('pending'); ?>：<strong style="color: #f0ad4e;"><?php echo number_format($comment_count->moderated); ?></strong></li>
                <li><?php echo occ_t('spam'); ?>：<strong style="color: #d9534f;"><?php echo number_format($comment_count->spam); ?></strong></li>
                <li><?php echo occ_t('trash'); ?>：<strong style="color: #868686;"><?php echo number_format($comment_count->trash); ?></strong></li>
            </ul>
        </div>

        <!-- 单独清空选项 -->
        <div class="card" style="max-width: 600px; margin-top: 20px;">
            <h2><?php echo occ_t('clear_by_type'); ?></h2>
            <p><?php echo occ_t('select_type'); ?></p>
            
            <div style="margin-bottom: 15px;">
                <form method="post" action="" style="display: inline-block; margin-right: 10px;">
                    <?php wp_nonce_field('occ_clear_comments_nonce'); ?>
                    <input type="hidden" name="action" value="clear_pending">
                    <input type="submit" class="button" value="<?php echo occ_t('clear_pending'); ?> (<?php echo number_format($comment_count->moderated); ?>)" 
                           <?php echo $comment_count->moderated == 0 ? 'disabled' : ''; ?>
                           onclick="return confirm('<?php echo occ_t('confirm_pending'); ?>');">
                </form>

                <form method="post" action="" style="display: inline-block; margin-right: 10px;">
                    <?php wp_nonce_field('occ_clear_comments_nonce'); ?>
                    <input type="hidden" name="action" value="clear_spam">
                    <input type="submit" class="button" value="<?php echo occ_t('clear_spam'); ?> (<?php echo number_format($comment_count->spam); ?>)" 
                           <?php echo $comment_count->spam == 0 ? 'disabled' : ''; ?>
                           onclick="return confirm('<?php echo occ_t('confirm_spam'); ?>');">
                </form>

                <form method="post" action="" style="display: inline-block;">
                    <?php wp_nonce_field('occ_clear_comments_nonce'); ?>
                    <input type="hidden" name="action" value="clear_trash">
                    <input type="submit" class="button" value="<?php echo occ_t('clear_trash'); ?> (<?php echo number_format($comment_count->trash); ?>)" 
                           <?php echo $comment_count->trash == 0 ? 'disabled' : ''; ?>
                           onclick="return confirm('<?php echo occ_t('confirm_trash'); ?>');">
                </form>
            </div>
        </div>

        <!-- 一键清空所有未批准评论 -->
        <div class="card" style="max-width: 600px; margin-top: 20px; border-color: #f0ad4e;">
            <h2 style="color: #f0ad4e;"><?php echo occ_t('clear_unapproved'); ?></h2>
            <p><?php echo occ_t('clear_unapproved_desc'); ?></p>
            
            <?php 
            $unwanted_total = $comment_count->moderated + $comment_count->spam + $comment_count->trash;
            ?>
            
            <form method="post" action="">
                <?php wp_nonce_field('occ_clear_comments_nonce'); ?>
                <input type="hidden" name="action" value="clear_unwanted">
                <p>
                    <input type="submit" class="button button-large" 
                           value="<?php echo occ_t('clear_all_unapproved'); ?> (<?php echo number_format($unwanted_total); ?>)" 
                           style="background-color: #f0ad4e; border-color: #f0ad4e; color: white;"
                           <?php echo $unwanted_total == 0 ? 'disabled' : ''; ?>
                           onclick="return confirm('<?php echo occ_t('confirm_unapproved'); ?>\n\n<?php echo occ_t('will_delete'); ?> <?php echo number_format($unwanted_total); ?> <?php echo occ_t('comments'); ?>');">
                </p>
            </form>
        </div>

        <!-- 危险操作：清空所有评论 -->
        <div class="card" style="max-width: 600px; margin-top: 20px; border-color: #dc3232;">
            <h2 style="color: #dc3232;"><?php echo occ_t('danger_clear_all'); ?></h2>
            <p><strong><?php echo occ_t('warning_all'); ?></strong></p>
            
            <form method="post" action="" onsubmit="return confirmDeleteAll();">
                <?php wp_nonce_field('occ_clear_comments_nonce'); ?>
                <input type="hidden" name="action" value="clear_all">
                <p>
                    <label>
                        <input type="checkbox" name="confirm_delete" id="confirm_delete" required>
                        <?php echo occ_t('confirm_delete_all'); ?> <?php echo number_format($total_comments); ?> <?php echo occ_t('comments'); ?>
                    </label>
                </p>
                <p>
                    <input type="submit" class="button button-primary" 
                           value="<?php echo occ_t('clear_all_comments'); ?>" 
                           style="background-color: #dc3232; border-color: #dc3232;"
                           <?php echo $total_comments == 0 ? 'disabled' : ''; ?>>
                </p>
            </form>
        </div>

        <div class="card" style="max-width: 600px; margin-top: 20px;">
            <h2><?php echo occ_t('instructions'); ?></h2>
            <ol>
                <li><strong><?php echo occ_t('clear_by_type'); ?></strong>：<?php echo occ_t('instruction_1'); ?></li>
                <li><strong><?php echo occ_t('clear_unapproved'); ?></strong>：<?php echo occ_t('instruction_2'); ?></li>
                <li><strong><?php echo occ_t('clear_all_comments'); ?></strong>：<?php echo occ_t('instruction_3'); ?></li>
                <li><?php echo occ_t('instruction_4'); ?></li>
                <li><?php echo occ_t('instruction_5'); ?></li>
            </ol>
        </div>
    </div>

    <script>
    function confirmDeleteAll() {
        var checkbox = document.getElementById('confirm_delete');
        if (!checkbox.checked) {
            return false;
        }
        return confirm('<?php echo occ_t('final_confirm'); ?> <?php echo number_format($total_comments); ?> <?php echo occ_t('cannot_undo'); ?>');
    }
    </script>
    <?php
}

// 按状态清空评论
function occ_clear_comments_by_status($status) {
    global $wpdb;
    
    $wpdb->query('START TRANSACTION');
    
    try {
        // 获取指定状态的评论ID
        $comment_ids = $wpdb->get_col($wpdb->prepare(
            "SELECT comment_ID FROM {$wpdb->comments} WHERE comment_approved = %s",
            $status
        ));
        
        if (!empty($comment_ids)) {
            $ids_string = implode(',', array_map('intval', $comment_ids));
            
            // 删除评论元数据
            $wpdb->query("DELETE FROM {$wpdb->commentmeta} WHERE comment_id IN ($ids_string)");
            
            // 删除评论
            $wpdb->query("DELETE FROM {$wpdb->comments} WHERE comment_ID IN ($ids_string)");
            
            // 更新相关文章的评论计数
            occ_update_comment_counts();
        }
        
        $wpdb->query('COMMIT');
        wp_cache_flush();
        
        return true;
    } catch (Exception $e) {
        $wpdb->query('ROLLBACK');
        return false;
    }
}

// 清空所有未批准评论（待审核、垃圾、回收站）
function occ_clear_unwanted_comments() {
    global $wpdb;
    
    $wpdb->query('START TRANSACTION');
    
    try {
        // 获取所有非批准状态的评论ID
        $comment_ids = $wpdb->get_col(
            "SELECT comment_ID FROM {$wpdb->comments} 
             WHERE comment_approved IN ('0', 'spam', 'trash')"
        );
        
        if (!empty($comment_ids)) {
            $ids_string = implode(',', array_map('intval', $comment_ids));
            
            // 删除评论元数据
            $wpdb->query("DELETE FROM {$wpdb->commentmeta} WHERE comment_id IN ($ids_string)");
            
            // 删除评论
            $wpdb->query("DELETE FROM {$wpdb->comments} WHERE comment_ID IN ($ids_string)");
            
            // 更新相关文章的评论计数
            occ_update_comment_counts();
        }
        
        $wpdb->query('COMMIT');
        wp_cache_flush();
        
        return true;
    } catch (Exception $e) {
        $wpdb->query('ROLLBACK');
        return false;
    }
}

// 清空所有评论
function occ_clear_all_comments() {
    global $wpdb;
    
    $wpdb->query('START TRANSACTION');
    
    try {
        // 删除所有评论元数据
        $wpdb->query("DELETE FROM {$wpdb->commentmeta}");
        
        // 删除所有评论
        $wpdb->query("DELETE FROM {$wpdb->comments}");
        
        // 重置所有文章的评论计数
        $wpdb->query("UPDATE {$wpdb->posts} SET comment_count = 0 WHERE post_type IN ('post', 'page')");
        
        $wpdb->query('COMMIT');
        wp_cache_flush();
        
        do_action('occ_comments_cleared');
        
        return true;
    } catch (Exception $e) {
        $wpdb->query('ROLLBACK');
        return false;
    }
}

// 更新评论计数
function occ_update_comment_counts() {
    global $wpdb;
    
    // 获取所有需要更新的文章ID
    $post_ids = $wpdb->get_col(
        "SELECT DISTINCT post_id FROM {$wpdb->posts} p
         WHERE p.post_type IN ('post', 'page') AND p.comment_count > 0"
    );
    
    foreach ($post_ids as $post_id) {
        wp_update_comment_count($post_id);
    }
}

// 添加设置链接到插件列表
add_filter('plugin_action_links_' . plugin_basename(__FILE__), 'occ_add_action_links');

function occ_add_action_links($links) {
    $settings_link = '<a href="admin.php?page=one-click-clear-comments">' . occ_t('menu_title') . '</a>';
    array_unshift($links, $settings_link);
    return $links;
}

// 激活插件时的操作
register_activation_hook(__FILE__, 'occ_activate');

function occ_activate() {
    add_option('occ_plugin_version', '1.2.0');
}

// 停用插件时的操作
register_deactivation_hook(__FILE__, 'occ_deactivate');

function occ_deactivate() {
    delete_option('occ_plugin_version');
    delete_option('occ_disable_all_comments');
}

// 禁用评论功能
add_action('init', 'occ_disable_comments_functionality');

function occ_disable_comments_functionality() {
    if (get_option('occ_disable_all_comments', 'no') === 'yes') {
        // 关闭所有文章类型的评论支持
        $post_types = get_post_types();
        foreach ($post_types as $post_type) {
            if (post_type_supports($post_type, 'comments')) {
                remove_post_type_support($post_type, 'comments');
                remove_post_type_support($post_type, 'trackbacks');
            }
        }
    }
}

// 禁用前台评论功能
add_action('template_redirect', 'occ_disable_comments_frontend');

function occ_disable_comments_frontend() {
    if (get_option('occ_disable_all_comments', 'no') === 'yes') {
        // 关闭评论
        add_filter('comments_open', '__return_false', 20, 2);
        add_filter('pings_open', '__return_false', 20, 2);
        
        // 隐藏现有评论
        add_filter('comments_array', '__return_empty_array', 10, 2);
        
        // 移除评论feed
        add_action('do_feed_rss2_comments', 'occ_disable_feed', 1);
        add_action('do_feed_atom_comments', 'occ_disable_feed', 1);
    }
}

// 禁用评论feed
function occ_disable_feed() {
    $message = array(
        'en' => 'Comments have been disabled.',
        'zh_CN' => '评论功能已被禁用。',
        'ja' => 'コメント機能は無効になっています。'
    );
    $lang = occ_get_current_language();
    wp_die($message[$lang]);
}

// 从管理栏和菜单中移除评论
add_action('admin_init', 'occ_disable_comments_admin');

function occ_disable_comments_admin() {
    if (get_option('occ_disable_all_comments', 'no') === 'yes') {
        // 重定向评论页面
        global $pagenow;
        if ($pagenow === 'edit-comments.php' && !isset($_GET['page'])) {
            wp_redirect(admin_url());
            exit;
        }
        
        // 移除评论菜单
        remove_menu_page('edit-comments.php');
        
        // 移除评论相关的管理栏项目
        add_action('wp_before_admin_bar_render', 'occ_remove_comments_admin_bar');
        
        // 移除仪表盘的评论小工具
        remove_meta_box('dashboard_recent_comments', 'dashboard', 'normal');
        
        // 从文章和页面编辑界面移除评论相关
        add_action('add_meta_boxes', 'occ_remove_comment_meta_boxes', 99);
    }
}

// 移除管理栏中的评论
function occ_remove_comments_admin_bar() {
    global $wp_admin_bar;
    $wp_admin_bar->remove_menu('comments');
}

// 移除评论元框
function occ_remove_comment_meta_boxes() {
    remove_meta_box('commentstatusdiv', 'post', 'normal');
    remove_meta_box('commentsdiv', 'post', 'normal');
    remove_meta_box('commentstatusdiv', 'page', 'normal');
    remove_meta_box('commentsdiv', 'page', 'normal');
}